# Loop‑Fluctuation Correlation Simulation (8×8)

This repository contains a self‑contained pipeline for computing correlations between per‑link **flip counts** and local **Wilson‑loop fluctuations** on an 8×8 lattice.  It extends the original Volume‑4 pipeline to arbitrary lattice size, adds deterministic gauge‑field generation from flip counts using a normalised angle transform (rather than the saturating logistic transform used previously), and provides tooling to bootstrap confidence intervals and produce a report summarising the results.

## Structure

```
vol4-loop-fluctuation-sim/
├── config.yaml               # YAML configuration for the pipeline
├── data/                     # input and generated data
│   ├── kernel_builder/       # base Hamiltonian kernel (real or dummy)
│   │   └── kernel.npy        # 1‑D kernel of length 2*L^2 (populated during setup)
│   ├── pivot_params.txt      # logistic transform parameters (a, b, logistic_k, logistic_n0)
│   ├── flip_counts.npy       # generated tick‑flip counts for each link (created by generate_flip_counts.py)
│   └── gauge_configs/        # gauge configurations per gauge group (created by sample_gauge_fields.py)
├── scripts/
│   ├── generate_flip_counts.py   # generate tick‑flip counts via tick‑flip operator algebra
│   ├── sample_gauge_fields.py    # build gauge links from flip counts and pivot params (instrumented with debug prints)
│   ├── run_correlation.py        # compute Wilson‑loop variances and correlations (instrumented with debug prints)
│   ├── generate_report.py        # produce bar plots and a Markdown summary of results
│   └── make_kernel_from_eigs.py  # optional helper to construct kernel.npy from eigenvalue spectra
├── requirements.txt          # Python dependencies
└── results/                  # output correlation table, figures and report
```

## Usage

1. **Install dependencies** (optional – the provided environment includes them):

   ```bash
   pip install -r requirements.txt
   ```

2. **Generate flip counts** for an 8×8 lattice:

   ```bash
   python scripts/generate_flip_counts.py --lattice-size 8 --seed 0 --output data/flip_counts.npy
   ```

3. **Sample gauge fields** using a normalised flip‑count transform:

   ```bash
   for G in U1 SU2 SU3; do
     python scripts/sample_gauge_fields.py \
       --flip-counts data/flip_counts.npy \
       --kernel data/kernel.npy \
       --pivot-config data/pivot_params.txt \
       --lattice-size 8 \
       --gauge-group "$G" \
       --trials 50 \
       --output-dir data/gauge_configs
   done
   ```

   The sampler prints debug information about the kernel location and shape.  Internally it maps each flip count to an angle between 0 and pi by normalising against the maximum count.  Gauge links for SU(2) are realised as 2×2 rotation matrices, while SU(3) links are constructed as block‑diagonal matrices with an SU(2) rotation in the top‑left block and a U(1) phase in the bottom‑right element.  A small deterministic perturbation is applied to each angle to generate multiple configurations.  Output files are saved as `.npy` arrays under `data/gauge_configs/`.

4. **Run the correlation analysis** and generate plots and a report:

   ```bash
   python scripts/run_correlation.py
   python scripts/generate_report.py
   ```

   The correlation table will be written to `results/correlation_full.csv`; plots and the Markdown report will be placed under `results/`.

## Notes

* The `sample_gauge_fields.py` script includes debug instrumentation to print whether a real Hamiltonian kernel file exists under `kernel_builder/kernel.npy` and the shape of the loaded kernel.  In this fixed version the kernel is only used for debugging; gauge links are generated deterministically by mapping flip counts to angles and constructing group‑valued matrices as described above.
* The pivot parameters listed in `data/pivot_params.txt` are retained for compatibility with the upstream implementation but are not used by the sampler.
* The `run_correlation.py` script prints a small sample of a loaded gauge configuration to verify that generated gauge links are non‑trivial matrices.  After running the full pipeline on an 8×8 lattice you should observe positive Pearson r values for both SU(2) and SU(3), with confidence intervals that exclude zero (see `results/correlation_full.csv`).
